#!/usr/bin/env python3

# gitlab-libvirt-executor - run GitLab-CI jobs within libvirt VMs
#
# Copyright © 2024, IOhannes m zmölnig, forum::für::umläute
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Affero General Public License as
#  published by the Free Software Foundation, either version 3 of the
#  License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Affero General Public License for more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import subprocess

from contextlib import contextmanager
import logging as _logging

_log = _logging.getLogger("gitlab-libvirt-executor.ssh")


@contextmanager
def SSH(*args, **kwargs):
    import paramiko

    class IgnorePolicy(paramiko.client.MissingHostKeyPolicy):
        """
        Policy that silently ignores any missing host keys.
        """

        def missing_host_key(self, client, hostname, key):
            _log.debug(
                "Unknown {} host key for {}: {}".format(
                    key.get_name(),
                    hostname,
                    paramiko.client.hexlify(key.get_fingerprint()),
                )
            )

    try:
        client = paramiko.SSHClient()
        client.load_system_host_keys()
        client.set_missing_host_key_policy(IgnorePolicy)
        client.connect(*args, **kwargs)
        yield client
    finally:
        client.close()


def checkSSH(
    host: str,
    username: str | None = None,
    password: str | None = None,
    key_filename: str | list[str] | None = None,
    timeout: float | None = None,
) -> bool:
    """test whether we can open an SSH connection to <host>"""
    try:
        with SSH(
            host,
            username=username,
            password=password,
            key_filename=key_filename,
            timeout=timeout,
        ) as client:
            pass
    except TimeoutError as e:
        _log.error("SSH connection to VM timed out!")
        _log.debug("ssh://%s@%s timed out with %s" % (username, host, e))
        return False
    except Exception as e:
        _log.exception("ssh://%s@%s failed with %s" % (username, host, e))
        return
    return True


def channel2file(inchan, outfile):
    """write data received from <inchan> to <outfile>"""
    while inchan.recv_ready():
        outfile.write(inchan.recv(65536))


def getbashexports(env={}):
    if not env:
        return b""
    cmd = subprocess.run(
        ["bash", "-c", "set"],
        stdout=subprocess.PIPE,
        env={str(k): str(v) for k, v in env.items()},
    )
    result = []
    for line in cmd.stdout.splitlines():
        k = line.split(b"=", maxsplit=1)
        if k[0].decode() in env:
            result.append(b"export " + line)
    return b"\n".join(result)


def run_commands(host, commands, **kwargs):
    """execute (on <host>) the given <script> (found on the hypervisor).
    <kwargs> are passed to SSH.
    """

    ret = None
    with (
        open("/dev/stdout", "wb", buffering=0) as binout,
        open("/dev/stderr", "wb", buffering=0) as binerr,
        SSH(host, **kwargs) as client,
    ):
        stdin, stdout, stderr = client.exec_command("/bin/bash", get_pty=False)
        stdin.write(commands)
        while not stdout.channel.exit_status_ready():
            channel2file(stderr.channel, binerr)
            channel2file(stdout.channel, binout)
        channel2file(stderr.channel, binerr)
        channel2file(stdout.channel, binout)
        ret = stdout.channel.exit_status

    return ret


def run_script(host, script, env={}, **kwargs):
    """copy script to <host> and execute it; possibly sourcing a file containing the <env> vars first"""
    dirname = "gitlab-%s" % (os.getpid())
    scriptname = os.path.join(dirname, "script")
    envname = os.path.join(dirname, "env")

    cmd = f"""#!/usr/bin/env bash
if set -o | grep pipefail > /dev/null; then set -o pipefail; fi; set -o errexit
set +o noclobber
: | eval $'source {envname}\n{scriptname}\n'
exit 0
"""

    ret = None
    with (
        open(script) as f,
        open("/dev/stdout", "wb", buffering=0) as binout,
        open("/dev/stderr", "wb", buffering=0) as binerr,
        SSH(host, **kwargs) as client,
    ):
        sftp = client.open_sftp()
        try:
            sftp.mkdir(dirname)
            sftp.putfo(f, scriptname)
            sftp.chmod(scriptname, 0o711)
            with sftp.open(envname, "wb") as envfd:
                envfd.write(getbashexports(env))

            stdin, stdout, stderr = client.exec_command(
                "/bin/bash", get_pty=False, environment=env
            )
            stdin.write(cmd)
            while not stdout.channel.exit_status_ready():
                channel2file(stderr.channel, binerr)
                channel2file(stdout.channel, binout)
            channel2file(stderr.channel, binerr)
            channel2file(stdout.channel, binout)
            ret = stdout.channel.exit_status
        finally:
            try:
                sftp.unlink(scriptname)
                sftp.unlink(envname)
                sftp.rmdir(dirname)
            except Exception:
                log.debug("Failed to cleanup %r on %r: %s" % (dirname, host, e))

    return ret


if __name__ == "__main__":
    import argparse
    from urllib.parse import urlparse

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--timeout",
        type=float,
        help="timeout for SSH-connection (DEFAULT: no timeout)",
    )
    parser.add_argument(
        "URI",
        type=urlparse,
        help="SSH-URI to check for connectivity (e.g. 'ssh://user:password@example.com')",
    )
    args = parser.parse_args()

    uri = args.URI
    if uri.scheme != "ssh":
        parser.exit(1, "URI-schema must be 'ssh'")

    if checkSSH(uri.hostname, uri.username, uri.password):
        parser.exit(0, "SSH connection to %r OK.\n" % (uri.hostname,))
    else:
        raise SystemExit(
            "couldn't establish an SSH connection to %r.\n" % (uri.hostname,)
        )
