#!/usr/bin/env python3

# gitlab-libvirt-executor - run GitLab-CI jobs within libvirt VMs
#
# Copyright © 2024, IOhannes m zmölnig, forum::für::umläute
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Affero General Public License as
#  published by the Free Software Foundation, either version 3 of the
#  License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Affero General Public License for more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

_name = "gitlab-libvirt-executor"
_version = None


import os
import time

try:
    from . import logger as logging
    from .namespace import Namespace
    from . import ssh
    from .virsh import virSH
    from . import clonefast
except ImportError:
    import logger as logging
    from namespace import Namespace
    import ssh
    from virsh import virSH
    import clonefast

if _version is None:
    try:
        try:
            from ._version import version as _version
        except ImportError:
            from _version import version as _version
    except ModuleNotFoundError:
        pass


defaults = Namespace(
    username="vagrant",
    password="vagrant",
    timeout=30,
    verbosity=0,
)


log = logging.getLogger(_name)


BUILD_FAILURE = 1
SYSTEM_FAILURE = 2

custom_env = {}


def setup():
    global BUILD_FAILURE
    global SYSTEM_FAILURE
    global custom_env

    BUILD_FAILURE = int(os.environ.get("BUILD_FAILURE_EXIT_CODE", BUILD_FAILURE))
    SYSTEM_FAILURE = int(os.environ.get("SYSTEM_FAILURE_EXIT_CODE", SYSTEM_FAILURE))

    prefix = "CUSTOM_ENV_"
    custom_env = {
        k[len(prefix) :]: v for k, v in os.environ.items() if k.startswith(prefix)
    }


def getVersion(libvirtURI=None):
    """get the version of the executor
    (possibly including libvirt version)
    """

    def parseVersion(ver):
        major = ver // 1000000
        minor = (ver % 1000000) // 1000
        release = ver % 1000
        return (major, minor, release)

    try:
        import libvirt

        with libvirt.openReadOnly(libvirtURI) as conn:
            typ = conn.getType()
            libver = {
                "libvirt": parseVersion(conn.getLibVersion()),
                typ: parseVersion(conn.getVersion()),
            }
            libvirt_version = ", ".join(
                f"{t}: {v0}.{v1}.{v2}" for t, (v0, v1, v2) in libver.items()
            )

    except Exception as e:
        log.debug("Unable to query libvirt for version", exc_info=True)
        libvirt_version = None

    ver = ""
    if _version:
        ver = _version
    if libvirt_version:
        ver = "%s (%s)" % (ver, libvirt_version)
    ver = ver.strip()
    if ver:
        return ver.strip()


def getVMimage() -> str:
    if not custom_env:
        setup()
    env = "CI_JOB_IMAGE"
    img = custom_env.get(env)
    if not img:
        log.fatal("no VM defined via %r" % (env,))
    return img


def getVMname() -> str:
    if not custom_env:
        setup()
    env = "CI_JOB_ID"
    job = custom_env.get(env)
    if job:
        return "gitlab-%s" % (job,)
    log.fatal("no GitlabEnv %r found" % (env,))


def getVMaddress(
    name: str | None = None,
    timeout: float | None = None,
    URI: str | None = None,
    protocol: str = "IPv4",
) -> str | None:
    if not name:
        name = getVMname()
    proto = None
    if protocol:
        proto = protocol.lower()

    virsh = virSH(name, URI=URI)

    start = time.time()
    now = start

    data = None
    while int(now - start) <= timeout:
        try:
            data = virsh.domifaddr()
            if data:
                break
        except:
            log.debug(
                "failed getting %s address for %r" % (protocol, name), exc_info=True
            )

        time.sleep(0.5)
        if timeout is not None:
            now = time.time()

    if not data:
        return

    for dev, ifconf in data.items():
        if proto and ifconf.get("protocol").lower() != proto:
            continue
        IP = ifconf.get("address")
        if IP:
            log.debug(
                "VM %r has %s %s (after %s seconds)"
                % (name, protocol, IP, time.time() - start)
            )
            return IP


def checkVMOnline(
    vmname: str,
    username: str | None = None,
    password: str | None = None,
    key_filename: str | list[str] | None = None,
    timeout: float | None = None,
    libvirtURI: str | None = None,
) -> str | None:
    """check if the given VM can be accessed with ssh://username:password@...
    on success, returns the IP of the VM; otherwise returns None
    """
    try:
        log.info("Waiting for the VM to become online...")
        IP = getVMaddress(vmname, timeout=timeout, URI=libvirtURI)
    except Exception as e:
        log.exception("failed to get IP address for %r" % (vmname,))
        return None

    if not IP:
        log.exception("couldn't get IP address for %r" % (vmname,))
        return None

    if username:
        # check whether the host is SSH-able
        log.info("Waiting for the VM to become SSH-able...")
        if not ssh.checkSSH(
            IP,
            username=username,
            password=password,
            key_filename=key_filename,
            timeout=timeout,
        ):
            return None
        log.info("Was able to SSH!")
        log.debug("ssh://%s@%s success!" % (username, IP))

    log.info("VM is ready.")
    return IP


def getAuth(args: Namespace) -> dict:
    """get dict with authentication information from args"""
    auth = {
        "username": None,
        "password": None,
        "key_filename": None,
    }
    try:
        auth["username"] = args.username
    except AttributeError:
        pass
    try:
        auth["password"] = args.password
    except AttributeError:
        pass
    try:
        keypath = args.identity_file
        if keypath:
            if os.path.isdir(keypath):
                # list files in keypath (https://stackoverflow.com/a/3207973/1169096)
                keypath = next(os.walk(keypath), (None, None, []))[2]
            auth["key_filename"] = keypath
    except AttributeError:
        pass
    return auth


def do_logtest(args):
    """print at various loglevels"""
    for name, level in logging.getLevelNamesMapping().items():
        log.log(level, name)

    print("log-level: %s/%s" % (log.level, log.getEffectiveLevel()))
    print(args)


def do_config(args):
    """configure subsequent stages of GitLab runner the runner"""
    import json
    import sys

    driver = {
        "name": _name or "gitlab-libvirt-executor",
    }
    ver = getVersion(args.connect)
    if ver:
        driver["version"] = ver

    try:
        builds_dir = args.guest_builds_dir
    except AttributeError:
        pass
    try:
        cache_dir = args.guest_cache_dir
    except AttributeError:
        pass

    job_env = {}

    # build the config dictionary
    localvars = locals()

    def makeConfig(*args):
        result = {}
        for k in args:
            v = localvars.get(k)
            if v:
                result[k] = v
        return result

    data = makeConfig(
        "driver",
        "job_env",
        "builds_dir",
        "cache_dir",
    )

    json.dump(data, sys.stdout)


def do_prepare(args):
    """create a VM gitlab-${CI_JOB_ID} and wait till it is accessible via SSH"""
    baseVM = None
    cloneVM = None
    auth = getAuth(args)
    timeout = args.timeout
    libvirtURI = args.connect
    outdir = args.clone_directory

    if outdir and not os.path.exists(outdir):
        # make sure that 'outdir' points to a directory
        outdir = os.path.join(outdir, "")

    if not custom_env:
        setup()

    shownbase = baseVM = custom_env.get("CI_JOB_IMAGE")
    if args.image and args.image != baseVM:
        shownbase = f"{baseVM!r} ({args.image!r})"
        baseVM = args.image

    if not baseVM:
        raise SystemExit("no base image given")
    log.info("Pulling the latest version of %s..." % (shownbase,))

    cloneVM = getVMname()
    if not cloneVM:
        raise SystemExit("unable to get name of cloned VM")
    log.info("Cloning a new VM...%r" % (cloneVM,))

    # start VM
    try:
        clonefast.clonefast(
            srcname=baseVM,
            dstname=cloneVM,
            startmode=clonefast.START_EPHEMERAL,
            outdir=outdir,
            libvirtURI=libvirtURI,
            quiet=True,
            omit_backingstore=args.omit_backingstore,
        )
    except Exception as e:
        log.exception(e)
        return SYSTEM_FAILURE

    # now wait until we have an SSH-connection
    host = checkVMOnline(
        cloneVM,
        timeout=timeout,
        libvirtURI=libvirtURI,
        **auth,
    )
    if not host:
        return SYSTEM_FAILURE

    if args.prepare_script:
        log.info("Running prepare script")
        try:
            with open(args.prepare_script) as f:
                cmds = f.read()
            ret = ssh.run_script(
                host,
                args.prepare_script,
                env=custom_env,
                **auth,
            )
            if ret:
                log.error("prepare script returned %s" % (ret,))
                do_cleanup(args)
                return BUILD_FAILURE
        except Exception as e:
            log.exception("failed to run prepare script")
            do_cleanup(args)
            return BUILD_FAILURE


def do_run(args):
    """run a script within the VM"""
    timeout = 30
    libvirtURI = args.connect

    vmname = getVMname()
    auth = getAuth(args)

    if args.action:
        log.debug("Running %r" % args.action)
    else:
        log.warning("Running unknown action")

    if not vmname:
        raise SystemExit("unable to get name of cloned VM")
    try:
        host = getVMaddress(vmname, timeout=timeout, URI=libvirtURI)
    except Exception as e:
        log.exception("failed to get IP address for %r" % (vmname,))
        return False

    try:
        with open(args.script) as f:
            cmds = f.read()
        ret = ssh.run_commands(host, cmds, **auth)
        if ret:
            log.error("%s script returned %s" % (args.action, ret))
            return BUILD_FAILURE
    except Exception as e:
        log.exception("failed to run %s script" % (args.action,))
        return BUILD_FAILURE
    return ret


def do_cleanup(args):
    """cleanup the VM"""
    cloneVM = getVMname()

    if not cloneVM:
        raise SystemExit("unable to get name of cloned VM")

    virsh = virSH(cloneVM, URI=args.connect)

    log.info("destroy VM %r" % (cloneVM,))
    if not virsh.destroy():
        raise SystemExit("couldn't destroy VM %r" % (cloneVM,))


def parseArgs():
    import argparse

    parser = argparse.ArgumentParser()
    parser.set_defaults(verbose=0, quiet=0)

    subparsers = parser.add_subparsers(required=True)

    p = subparsers.add_parser("logtest", help="Test logging with verbosity")
    p.set_defaults(func=do_logtest, verbose=0, quiet=0)
    p.add_argument(
        "-v",
        "--verbose",
        action="count",
        help="raise verbosity (can be given multiple times",
    )
    p.add_argument(
        "-q",
        "--quiet",
        action="count",
        help="lower verbosity (can be given multiple times",
    )

    p = subparsers.add_parser("config", help="Configure GitLab Runner")
    p.set_defaults(func=do_config)

    p = subparsers.add_parser("prepare", help="Prepare a libvirt VM for execution")
    p.set_defaults(func=do_prepare)
    p.add_argument(
        "--timeout",
        type=int,
        help="libvirt connection timeout for VM boot and SSH connection (DEFAULT: %r)"
        % defaults.timeout,
    )
    p.add_argument(
        "--prepare-script",
        help="path to script (on hypervisor) that is executed within the VM to complete the preparation",
    )
    p.add_argument(
        "--clone-directory",
        help="directory for storing disk images of (ephemeral) clone VMs",
    )
    p.add_argument(
        "--omit-backingstore",
        default=None,
        action="store_true",
        help="Do not create a <backingStore/> tag in the new guest definition (might be needed for older libvirt versions)",
    )

    p = subparsers.add_parser("run", help="Run GitLab's scripts in libvirt VM")
    p.set_defaults(func=do_run)
    p.add_argument(
        "script",
        help="script file to execute",
    )
    p.add_argument(
        "action",
        help="Name of the action being executed",
    )

    p = subparsers.add_parser("cleanup", help="Cleanup libvirt VM after job finishes")
    p.set_defaults(func=do_cleanup)

    p = parser.add_argument_group("SSH options")
    p.add_argument(
        "-u",
        "--username",
        help="SSH user (DEFAULT: %r)" % (defaults.username,),
    )
    p.add_argument(
        "-p",
        "--password",
        help="SSH password",
    )
    p.add_argument(
        "--identity-file",
        help="SSH identity file for authentication (if a password is required, uses '--password')",
    )

    p = parser.add_argument_group("libvirt options")
    p.add_argument(
        "--connect",
        help="libvirt connection URI",
    )

    p = parser.add_argument_group("configuration")
    p.add_argument(
        "--config",
        action="append",
        help="additional configuration-file to use",
    )
    p.add_argument(
        "--exclude-unconfigured",
        default=None,
        action="store_true",
        help="only allow starting of VMs that are explicitly configured in the configuration files (DEFAULT)",
    )
    p.add_argument(
        "--include-unconfigured",
        default=None,
        dest="exclude_unconfigured",
        action="store_false",
        help="allow starting of VMs that are not explicitly configured in the configuration files",
    )

    p = parser.add_argument_group("verbosity")
    p.add_argument(
        "-v",
        "--verbose",
        action="count",
        help="raise verbosity (can be given multiple times)",
    )
    p.add_argument(
        "-q",
        "--quiet",
        action="count",
        help="lower verbosity (can be given multiple times)",
    )
    p.add_argument(
        "--no-run",
        action="store_true",
        help="don't actually run (exit early)",
    )

    args = parser.parse_args()

    # merge verbosity settings
    if args.verbose or args.quiet:
        args.verbosity = args.verbose - args.quiet
        verbosity = args.verbosity
    else:
        args.verbosity = None
        verbosity = defaults.verbosity
    logging.setVerbosity(verbosity)
    del args.quiet
    del args.verbose

    return args


def parseConfigs(configfiles, **kwargs):
    """parse configuration file(s)
    any given keyword arguments override settings in the configuration files
    """
    import configparser

    # general options
    # - libvirt URI
    # - specify VM clone/start command
    # - only allow images that have a configuration section
    # per VM options
    # - ssh credentials
    # - cache/build directories (currently not implemented)
    example = """
[DEFAULT]
# general options
libvirtURI = qemu:///system
exclude-unconfigured = true

# VM-defaults
username = vagrant
password = vagrant
guest-builds-dir "/path/on/guest/builds"
guest-cache-dir "/path/on/guest/cache"
builds-dir "/path/on/host/builds"
cache-dir "/path/on/host/cache"


# VM overrides (per VM-name)
[win10]
    username = admin
    password = secret
    """
    default_configfiles = [
        "/etc/gitlab-runner/gitlab-libvirt-executor.ini",
        os.path.expanduser("~/.config/gitlab-runner/gitlab-libvirt-executor.ini"),
    ]

    cfg = configparser.ConfigParser()
    did_configfiles = cfg.read(configfiles or default_configfiles)
    if not did_configfiles:
        log.warning("No configuration files found!")

    globaloptions = ["exclude-unconfigured"]
    getters = {
        "exclude-unconfigured": "boolean",
        "omit-backingstore": "boolean",
        "timeout": "float",
        "verbosity": "int",
    }

    def section2dict(section, options=None, excluded_options=[]):
        result = {}
        if options is None:
            options = cfg.options(section)
        for option in options:
            if option in excluded_options:
                continue
            getter = getattr(cfg, "get" + getters.get(option, ""))
            result[option.replace("-", "_")] = getter(section, option)
        return result

    data = {}
    for section in cfg:
        if section == cfg.default_section:
            continue
        sectiondata = section2dict(section, excluded_options=globaloptions)
        sectiondata.update(kwargs)
        if sectiondata:
            data[section] = sectiondata

    # resolve 'extends'
    extends = {}
    for name, section in data.items():
        if "extends" in section:
            x = section["extends"]
            if x == name:
                raise SystemExit(f"VM {name!r} cannot extend itself")
            if x not in data:
                raise SystemExit(f"VM {name!r} extends unknown VM {x!r}")
            extends[name] = section["extends"]
            del section["extends"]

    def dict_setdefaults(d: dict, defaults: dict):
        """update <d> with defaults from <dict>"""
        for k, v in defaults.items():
            d.setdefault(k, v)

    images = {}
    while extends:
        for vm in list(extends):
            x = extends[vm]
            if x == vm:
                raise SystemExit(f"VM {vm!r} transitively extends itself")
            dict_setdefaults(data[vm], data[x])
            if x in extends:
                extends[vm] = extends[x]
            else:
                del extends[vm]
            images[vm] = data[x].get("image", x)

    for vm, img in images.items():
        data[vm].setdefault("image", img)

    defaultcfg = section2dict(cfg.default_section, options=cfg.defaults())
    defaultcfg.update(kwargs)
    if not defaultcfg.get("exclude_unconfigured", True):
        try:
            del defaultcfg["exclude_unconfigured"]
        except KeyError:
            pass
        data[None] = defaultcfg

    return data


def getConfiguration():
    args = parseArgs()

    fun = args.func
    del args.func

    configfiles = args.config
    del args.config

    log.debug(args)
    cfg_override = {k: v for k, v in args.__dict__.items() if v is not None}

    cfg = parseConfigs(configfiles, **cfg_override)
    log.debug(cfg)

    img = getVMimage()

    imgcfg = None
    if img:
        if img in cfg:
            imgcfg = Namespace(**cfg[img])
        elif None in cfg:
            imgcfg = Namespace(**cfg[None])
        else:
            log.fatal(
                "Building with image %r is not allowed by the administrator." % (img,)
            )
            if fun in [
                do_prepare,
            ]:
                raise SystemExit(BUILD_FAILURE)
            imgcfg = Namespace(**cfg_override)
    if imgcfg is None:
        imgcfg = Namespace(**cfg_override)

    imgcfg.setdefaults(defaults)
    log.debug(imgcfg)
    if imgcfg.verbosity is not None:
        logging.setVerbosity(imgcfg.verbosity)

    return (fun, imgcfg)


def _main():
    setup()
    try:
        fun, args = getConfiguration()
    except SystemExit as e:
        if e.code and type(e.code) != int:
            log.fatal(e)
            raise SystemExit(SYSTEM_FAILURE)
        raise (e)

    if args.no_run:
        raise SystemExit(0)

    ret = None
    try:
        ret = fun(args)
    except SystemExit as e:
        if e.code and type(e.code) != int:
            log.fatal(e)
            raise SystemExit(SYSTEM_FAILURE)
        raise (e)
    raise SystemExit(ret)


if __name__ == "__main__":
    _main()
