#!/usr/bin/env python3

# virt-clonefast - clone VMs fast
#
# Copyright © 2024, IOhannes m zmölnig, forum::für::umläute
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Affero General Public License as
#  published by the Free Software Foundation, either version 3 of the
#  License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Affero General Public License for more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# this creates shallow clones of an existing VM
# - qcow2 disks are duplicated as shallow
# - CDROMS are duplicated as is
# - any remaining disks throw an error

import sys
import os
from xml.dom import minidom

import pprint
import logging
import libvirt

if __name__ == "__main__":
    log = logging.getLogger("virt-clonefast")
else:
    log = logging.getLogger("gitlab-libvirt-executor.clonefast")

logging.basicConfig()

START_PRINTXML = 0
START_DEFINE = 1
START_EPHEMERAL = 2
START_PERSISTENT = 3


def _libvirt_silent_error_handler(ctx, err):
    """suppress libvirt errors"""
    return 1


def parseArgs():
    """parse commandline arguments"""
    import argparse

    parser = argparse.ArgumentParser(
        description="""Duplicate a virtual machine, changing all the unique host side configuration like MAC address, name, etc.""",
        epilog="""This is very similar to 'virt-clone(1)', but uses shallow clones for copy-on-write disks.
Currently, only QCOW2 disks are supported.
    """,
    )
    parser.set_defaults(start=START_DEFINE)

    parser.add_argument(
        "--connect",
        metavar="URI",
        help="Connect to hypervisor with libvirt URI (note that to-be-cloned disks must be locally available)",
    )

    parser.add_argument(
        "name",
        metavar="NEW_NAME",
        nargs="?",
        help="Name for the new guest (overrides the '--name' argument)",
    )

    g = parser.add_argument_group("General Options")
    g.add_argument(
        "-o",
        "--original",
        metavar="SRC_NAME",
        required=True,
        help="Name of the original guest to clone.",
    )
    g.add_argument(
        "-n",
        "--name",
        dest="opt_name",
        metavar="NEW_NAME",
        help="Name for the new guest",
    )

    g = parser.add_argument_group("Clone Options")
    g.add_argument(
        "-m",
        "--mac",
        metavar="NEW_MAC",
        help="New fixed MAC address for the clone guest. Default is a randomly generated MAC.",
    )

    g = parser.add_argument_group("Output Options")
    g.add_argument(
        "--outdir",
        help="Directory to put cloned disk images into",
    )
    g = parser.add_argument_group("Output Options")
    g.add_argument(
        "--omit-backingstore",
        action="store_true",
        help="Do not create a <backingStore/> tag in the new guest definition (might be needed for older libvirt versions)",
    )
    g = parser.add_mutually_exclusive_group()
    g.add_argument(
        "--define",
        dest="start",
        action="store_const",
        const=START_DEFINE,
        help="Define a new persistent VM. (DEFAULT)",
    )
    g.add_argument(
        "--print-xml",
        dest="start",
        action="store_const",
        const=START_PRINTXML,
        help="Print the generated domain XML rather than create the guest.",
    )
    g.add_argument(
        "--start-ephemeral",
        dest="start",
        action="store_const",
        const=START_EPHEMERAL,
        help="Start the cloned VM as an ephemeral domain (the VM and all it's disks will be destroyed once powered down).",
    )
    g.add_argument(
        "--start-persistent",
        dest="start",
        action="store_const",
        const=START_PERSISTENT,
        help="Define a new persistent VM and start it.",
    )

    g = parser.add_argument_group("Verbosity")
    g.add_argument(
        "-q",
        "--quiet",
        action="store_true",
        help="Suppress non-error output",
    )
    g.add_argument(
        "-d",
        "--debug",
        action="store_true",
        help="Print debugging information",
    )

    args = parser.parse_args()

    if not args.name:
        args.name = args.opt_name
    del args.opt_name

    # make sure that outdir is a directory (not a filename)
    if args.outdir is not None:
        args.outdir = os.path.join(args.outdir, "")

    if args.debug:
        log.setLevel(logging.DEBUG)

    return args


class Domain:
    def __init__(self, xml: str):
        self.xml = minidom.parseString(xml)

    def toXML(self):
        return self.xml.toxml()

    def __str__(self):
        return self.toXML()

    def getDisks(self) -> dict[str, str]:
        disks = []
        for diskType in self.xml.getElementsByTagName("disk"):
            disk = {}
            for diskNode in diskType.childNodes:
                name = diskNode.nodeName
                if name[0:1] == "#":
                    continue
                disk[name] = {
                    diskNode.attributes[attr].name: diskNode.attributes[attr].value
                    for attr in diskNode.attributes.keys()
                }
            disks.append(
                {
                    "type": diskType.getAttribute("type"),
                    "device": diskType.getAttribute("device"),
                    "file": disk.get("source", {}).get("file"),
                    "driver": disk.get("driver", {}).get("type"),
                    "target": disk.get("target", {}).get("dev"),
                }
            )
        return disks

    def changeDiskSourceFile(
        self,
        target_device: str,
        source_file: str,
        backing_file: str | None = None,
        format: str = "qcow2",
    ) -> bool:
        """changes the source file of <device> to <path>"""

        def addChild(parent, tagname):
            el = minidom.Element(tagname)
            parent.appendChild(el)
            el.ownerDocument = parent.ownerDocument
            return el

        for diskType in self.xml.getElementsByTagName("disk"):
            device = None
            for target in diskType.getElementsByTagName("target"):
                if target.getAttribute("dev") == target_device:
                    device = True
            if not device:
                continue
            for source in diskType.getElementsByTagName("source"):
                if source.getAttribute("file"):
                    source.attributes["file"].value = source_file
                # add backingstore if required
                if backing_file:
                    backingStore = source.parentNode.getElementsByTagName(
                        "backingStore"
                    )
                    if backingStore:
                        backingStore = backingStore[0]
                    else:
                        backingStore = addChild(source.parentNode, "backingStore")
                    for el in backingStore.childNodes:
                        backingStore.removeChild(el)
                    backingStore.setAttribute("type", "file")
                    addChild(backingStore, "format").setAttribute("type", format)
                    addChild(backingStore, "source").setAttribute("file", backing_file)

                return True
        return False

    def convertToClone(self, newname: str, mac_address: str | None = None) -> bool:
        """convert the Domain-description to a description of a clone
        this basically changed the name to <name>, and removes any 'uuid'"""
        domain = None
        for name in self.xml.getElementsByTagName("name"):
            try:
                if name.parentNode.parentNode != self.xml:
                    continue
            except AttributeError:
                continue

            for n in name.childNodes:
                log.debug("rename %r -> %r" % (n.data, newname))
                n.data = newname
                domain = name.parentNode
                break
            if domain:
                break
        if not domain:
            return False
        for uuid in domain.getElementsByTagName("uuid"):
            if uuid.parentNode != domain:
                continue
            domain.removeChild(uuid)

        for mac in domain.getElementsByTagName("mac"):
            if mac.parentNode.tagName != "interface":
                continue
            if mac.hasAttribute("address"):
                mac.removeAttribute("address")
            if mac_address:
                mac.setAttribute("address", mac_address)

        return True

    def getClonableDisks(self) -> dict:
        """check if all disks are either QCOW2-file-based disks or (implicitly sharable) CD-ROMs
        returns a dictionary of targetdev:qcow2image mappings
        exits if an unsupported device is found"""
        disks = {}
        for idx, d in enumerate(self.getDisks()):
            device = d.get("device")
            driver = d.get("driver")
            dtype = d.get("type")
            if device == "cdrom":
                continue
            if device != "disk":
                raise ValueError(
                    "Disk#%d is is an unsupported device '%s'" % (idx, device)
                )
            if dtype != "file":
                raise ValueError(
                    "Disk#%d is '%s'-based (only 'file' is supported)" % (idx, dtype)
                )
            # check if the disk is qcow2 based
            if driver != "qcow2":
                raise ValueError(
                    "Disk#%d is of type '%s' (only 'qcow2' is supported)"
                    % (idx, driver)
                )
            disks[d["target"]] = d["file"]
        return disks

    @classmethod
    def fromLibVirt(cls, conn: libvirt.virConnect, name: str) -> libvirt.virDomain:
        lookups = [conn.lookupByName, conn.lookupByUUIDString, conn.lookupByUUID]
        dom = None
        for lookup in lookups:
            try:
                dom = lookup(name)
                break
            except libvirt.libvirtError:
                pass
        if dom:
            try:
                return cls(dom.XMLDesc())
            except minidom.xml.parsers.expat.ExpatError:
                pass


def makeStoragePoolXML(name: str, path: str) -> str:
    """create an XML definition of a dir-based storage pool"""

    def addText(parent, text):
        txt = minidom.Text()
        txt.data = text
        parent.appendChild(txt)
        return txt

    raw_xml = """<pool type='dir'><name/><target><path/></target></pool>"""
    xml = minidom.parseString(raw_xml)
    for n in xml.getElementsByTagName("name"):
        addText(n, name)
        break
    for p in xml.getElementsByTagName("path"):
        addText(p, path)
        break
    return xml.toxml()


def makeStoragePool(
    conn: libvirt.virConnect, path: str
) -> libvirt.virStoragePool | None:
    import uuid

    pools = conn.listStoragePools()
    poolname = "tmp-%s" % uuid.uuid4()
    while poolname in pools:
        poolname = "tmp-%s" % uuid.uuid4()

    # found a poolname that is not yet taken
    try:
        pool = conn.storagePoolCreateXML(makeStoragePoolXML(poolname, path))
        log.debug("created (ephemeral) storage pool %r" % (pool.name))
        return pool
    except:
        # couldn't create pool, presumably because outdir is already in some other pool
        return


def rescanStoragePools(conn: libvirt.virConnect):
    """re-scan all storage pools of connection"""
    for p in conn.listAllStoragePools():
        log.debug("refresh storage pool %r" % (p.name()))
        p.refresh()


def cleanupStoragePool(pool: libvirt.virStoragePool):
    """destroy the ephemeral pool (if it is empty)"""
    if pool and not pool.listAllVolumes():
        log.debug("destroying ephemeral pool %r" % (pool.name(),))
        pool.destroy()


def cleanupDiskImages(conn: libvirt.virConnect, diskimages: list[str]):
    """delete volumes for ephemeral VMs"""
    images = set(diskimages)
    for p in conn.listAllStoragePools():
        for v in p.listAllVolumes():
            path = v.path()
            if path in images:
                log.debug("deleting ephemeral volume %r in %r" % (path, p.name()))
                v.delete()
                images.discard(path)
            if not images:
                break
        if not images:
            break


def cloneQCOW2(source: str, target: str | None = None) -> str:
    """copy-on-write clone of a QCOW2 file <source> to another QCOW2 file
    if target is None, the clone will be in the same directory as the source, and the name will be derived from the source basename.
    if target is a directory (or ends with "/"), the clone will be in the given directory, and the name will be derived from the source basename.
    otherwise, target is a the prospective name of the output file.
    cloneQCOW2 will refuse to overwrite existing images, and use a unique filename when required.
    the actual filename of the cloned image is returned
    """
    import subprocess

    # qemu-img create -f qcow2 -b base.qcow2 -F qcow2 clone.qcow2

    # check if source exists and can be opened (otherwise raise a standard error)
    with open(source) as f:
        pass

    if not target:
        target = source
    elif os.path.isdir(target) or not os.path.basename(target):
        target = os.path.join(target, os.path.basename(source))

    # ensure that output directory exists
    outdir = os.path.dirname(target)
    os.makedirs(outdir, exist_ok=True)

    base, ext = os.path.splitext(target)

    i = ""
    while True:
        target = "%s%s%s" % (base, i, ext)
        try:
            targetfd = open(target, "x")
            break
        except FileExistsError:
            pass
        if not i:
            i = 0
        i -= 1

    targetfd.close()

    log.debug("shallow-cloning %r to %r" % (source, target))

    # and duplicate the image
    subprocess.run(
        [
            "qemu-img",
            "create",
            "-q",
            "-f",
            "qcow2",  # target format
            "-b",
            source,  # backing file
            "-F",
            "qcow2",  # backing format
            target,
        ],
        check=True,
        stdout=sys.stderr,
    )

    return target


def cloneDisks(disks: dict[str, str], outputdir: str | None) -> dict[str, str]:
    """shallow-clones the given disks into output dir

    <disks> is a dictionary that maps IDs to disk paths.
    a similar map from IDs to cloned paths is returned
    """
    return {k: cloneQCOW2(v, outputdir) for k, v in disks.items()}


def clonefast(
    srcname: str,
    dstname: str,
    startmode: int = START_DEFINE,
    libvirtURI: str | None = None,
    mac: str | None = None,
    outdir: str | None = None,
    quiet: bool = False,
    omit_backingstore: bool = None,
):
    """do a fast clone of a VM
    <srcname>: original VM
    <dstname>: VM to be created
    <startmode>: how to start the newly created VM
    <libvirtURI>: connection URI to the libvirt server
        (note that only local QCOW2 disks can be cloned)
    <mac>: MAC-address of cloned network interface (use None for a new random MAC)
    <outdir>: place for the shallow cloned disks images (use None to use the same directory as the input disk images)
    <quiet>: if True, suppress libvirt errors

    <omit_backingstore>: if True, does not create a <backingStore/> entry for COW-images (might work around a bug with older libvirt)
      if None, tries to autodetect whether the <backingStore/> should be omitted
    """

    if quiet:
        libvirt.registerErrorHandler(_libvirt_silent_error_handler, None)

    libvirt_open = libvirt.open
    if startmode == START_PRINTXML:
        libvirt.openReadOnly

    with libvirt_open(libvirtURI) as conn:
        if omit_backingstore is None:
            omit_backingstore = conn.getLibVersion() < 9010000

        dom = Domain.fromLibVirt(conn, srcname)
        if not dom:
            raise KeyError("Couldn't find original VM %r" % (srcname,))

        if not dstname:
            infix = 0
            while True:
                if infix:
                    name = "%s-clone-%d" % (srcname, infix)
                else:
                    name = "%s-clone" % (srcname,)
                if not Domain.fromLibVirt(conn, name):
                    dstname = name
                    break
                infix += 1

        if Domain.fromLibVirt(conn, dstname):
            raise KeyError("cloned VM %r already exists" % (dstname,))

        dom.convertToClone(dstname, mac)

        srcdisks = dom.getClonableDisks()

        log.debug("cloning disks %s to %r" % (srcdisks, outdir))
        cloneddisks = cloneDisks(srcdisks, outdir)
        log.debug("cloned disks %s" % (cloneddisks,))

        pool = None
        if startmode == START_EPHEMERAL and outdir:
            pool = makeStoragePool(conn, outdir)

        # re-scan all the pools
        rescanStoragePools(conn)

        for k, v in cloneddisks.items():
            log.debug("fixing clone disk %r" % k)
            backdisk = srcdisks[k]
            if omit_backingstore:
                backdisk = None
            dom.changeDiskSourceFile(k, v, backdisk)

        if startmode == START_PRINTXML:
            print(dom)
        elif startmode == START_EPHEMERAL:
            log.debug("start ephemeral VM %r" % (dstname,))
            conn.createXML(dom.toXML())
            # delete volumes for ephemeral VM
            cleanupDiskImages(conn, cloneddisks.values())
            cleanupStoragePool(pool)
        else:
            log.debug("define cloned VM %r" % (dstname,))
            vm = conn.defineXML(dom.toXML())
            if startmode == START_PERSISTENT:
                log.debug("starting persistent VM %r" % (dstname,))
                vm.create()


def _main():
    args = parseArgs()
    log.debug(args)

    try:
        clonefast(
            args.original,
            args.name,
            startmode=args.start,
            libvirtURI=args.connect,
            mac=args.mac,
            outdir=args.outdir,
            quiet=args.quiet,
            omit_backingstore=args.omit_backingstore,
        )
    except Exception as e:
        raise SystemExit(e)


if __name__ == "__main__":
    _main()
