/*  Copyright (C) 2001-2002 Vanja Hrustic

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
    USA

    In addition, as a special exception, Vanja Hrustic
    gives permission to link the code of this program with
    the SAVI library, and distribute linked combinations including
    the two. You must obey the GNU General Public License in all
    respects for all of the code used other than SAVI. If you modify
    this file, you may extend this exception to your version of the
    file, but you are not obligated to do so. If you do not wish to
    do so, delete this exception statement from your version.

*/

#include "sophie.h"

/* $Id: sophie.c,v 1.7 2003/04/24 15:38:40 vanja Exp $ */

volatile int PROC_COUNT = 0;

char *default_configs[] = {
	"/etc/sophie.cfg",
	"/usr/local/etc/sophie.cfg",
	"./sophie.cfg",
	NULL
};

/* Configuration file */
char config_file[MAXPATHLEN];

/* working process pid */
pid_t wpid = 0;

/* child or not */
int is_child = 0;

/* Print the usage */
static void usage(void)
{
	fprintf(stdout, "\n(Version: %s)\n\n", SOPHIE_VERSION);
	fprintf(stdout, "USAGE: %s [options]\n", program_name);
	fprintf(stdout, "Options:\n");
	fprintf(stdout, "  -C [path] Sophie configuration file location\n");
	fprintf(stdout, "  -c        Show compiled-in Sophos/SAVI engine configuration\n");
	fprintf(stdout, "  -D        Daemon mode (forks into background)\n");
	fprintf(stdout, "  -d        Debug mode (shows debugging output, does not fork into background)\n");
	fprintf(stdout, "  -f [path] Filename (to scan for viruses)\n");
	fprintf(stdout, "  -s [path] Directory (to scan for viruses)\n");
	fprintf(stdout, "  -h        Help (this screen)\n");
	fprintf(stdout, "  -v        SAVI version (engine and patterns)\n");
}

void cleanup()
{
    sophie_print(0, "%s Cleanup socket(s) and pidfile", NOTESTR);
    close(sock);
#ifdef SOPHIE_NET
	shutdown(tcp_sock, 2);
	close(tcp_sock);
#endif
	unlink(config.socketfile);
	unlink(config.pidfile);   
}

/*
   We don't want any zombies
   Hope this is the right way to do it :)
*/
void sig_chld(int sig)
{
	/* Suggested by philipp@corpex.de (Philipp Gaschtz) */
	int stat;
	pid_t cpid;
	int serrno = errno;

	/* while there are any children that have exited... */
	while((cpid = waitpid(-1, &stat, WNOHANG) > 0) && PROC_COUNT)
	{
		if (cpid == 0) { } /* no children here */

		if (cpid == -1)
		{
			sophie_print(1, "%s There has been a problem catching a child: %s", WARNSTR, strerror(errno));
			break;
		}

		if (cpid > 0)
		{
			/* Should not happen, but... */
			if (PROC_COUNT > 0)
				PROC_COUNT--;
		}
	}

	errno = serrno;
}

/* Print PROC_COUNT */
void sig_usr1(int sig)
{
	sophie_print(0, "%s Current process count (PROC_COUNT): [%d]", NOTESTR, PROC_COUNT);
}

/* Cleanup and exit */
void sig_exit(int sig)
{	
	if (!is_child) {
		/* only call if not a child scan process */
		sophie_print(0, "%s SIGNAL '%d' caught - cleaning up and exiting.", NOTESTR, sig);	
		sophie_end();	
		exit(0);
	}
}

void sig_main_general(int sig)
{
	sophie_print(0, "%s SIGNAL '%d' caught - forwarding to working process", NOTESTR, sig);
	kill(wpid, sig);
}

void sig_pipe(int sig)
{
//	sophie_print(0, "SIGPIPE signal received - client probably disconnected");
}

/* Restart Sophie */
void sig_reload(int sig)
{
	HRESULT hr;
	int config_ret;

	sophie_print(0, "%s Reloading virus patterns/engine", NOTESTR);

	hr = pSAVI->pVtbl->LoadVirusData(pSAVI);
	if (SOPHOS_FAILED(hr))
	{
		sophie_print(0, "%s Unable to reload patterns/engine", ERRSTR);
		return;
	}

	if (hr == SOPHOS_SAVI_ERROR_OLD_VIRUS_DATA)
		sophie_print(0, "%s Virus data is out of date.", WARNSTR);

	sophie_print(0, "%s Virus patterns/engine reloaded", NOTESTR);

	/* Reload SAVI options */
	sophie_set_engine_config(pSAVI);

	/* Reload Sophie config */
	sophie_config_reset();
	config_ret = sophie_load_config(config_file);
	if (config_ret != 1)
		exit(EXIT_FAILURE);

	sophie_version();
}

/* Catch SIGALRM delivered to parent process - still no clue why it happens */
void sig_palarm(int sig)
{
	syslog(LOG_ERR, "Sophie parent process has just received SIGALRM - resuming!\n");
}

void sophie_timeout(int sig)
{
	syslog(LOG_ERR, "Sophie child has timed-out (no data received in %d seconds) - process killed\n", config.timeout);
	exit(-1);
}

/* Set socket options... */
void set_sockopts(int fd)
{
	int opt;

	sophie_print(1, "%s fd %d setting TCP_NODELAY", NOTESTR, fd);
	if (setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, &opt, sizeof opt) == -1)
		sophie_print(0, "%s setsockopt TCP_NODELAY: %.100s", WARNSTR, strerror(errno));

	sophie_print(1, "%s fd %d setting SO_RESUSEADDR", NOTESTR, fd);
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt)) == -1)
		sophie_print(0, "%s setsockopt SO_REUSEADDR: %s", NOTESTR, strerror(errno));
}

/* Here we go... */
int main(int argc, char *argv[])
{
	/* Result returned by SAVI */
	int sophos_result = 0;
	
	/* Related to socket */
	int msgsock = -99;
	
	/* Related to other stuff */
	int c, pid, cpid;
	
	/* 'Read' buffer */
	char buf[MAXPATHLEN];
	
	/* Pathfile to scan */
	char scan[MAXPATHLEN];

	/* Path (directory) to scan */
	char dirscan[MAXPATHLEN];

	/* Response to send back */
	char sock_response[256];
	
	/* fileInfo used to check if the socket is really a socket :) (for IS_SOCK()) */
	struct stat fileInfo;
	
	/* socket structure... */
	struct sockaddr_un server;
	
	/* Used when changing the group of the socket */
	struct group *grpInfo;

	/* Used to check if the user exists */
	struct passwd *checkUser;

	/* Stupid signal handling code */
	int sig_ret;
	sigset_t set;
	sigset_t bset; /* blockset, for fork() */
	struct sigaction sa_chld;
	struct sigaction sa_hup;
	struct sigaction sa_usr1;

	/* child */
	struct sigaction sa_handle;
	struct stat reqstat;
	struct passwd *userInfo;

	/* temporary file descriptor to detach when daemonizing */
	int devnull;

	/* PID file */
	FILE *f;

	/* Config result */
	int config_ret;

	/* Scan type (local/network) */
	int scan_type = 0;

#ifdef SOPHIE_NET
	/* Network socket related stuff */
	int client_tcp_sock = -99;
	int reuse_addr = 1;
	struct protoent *proto;
	struct sockaddr_in tcp_sock_data;

	struct sockaddr addr;
	socklen_t addrlen;
/*   int childpid, clilen; */
#endif

	struct timespec rqtp;
	rqtp.tv_sec = 0;
	rqtp.tv_nsec = 10000;

	/* Copy the name of the program (as supplied on the command line) */
	program_name = argv[0];
	program_args = argv;

    /* The main process now catches and executes sig_exit.  Upon the exit
       of the main process, the startup process (running as root) will
       unblock from its wait and cleanup the socket(s) and pidfile, etc */
    signal(SIGALRM, SIG_IGN);
/*	signal(SIGCHLD, sig_chld); */
/*	signal(SIGHUP, sig_reload); */
	signal(SIGINT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGSEGV, SIG_IGN);

	/* Clear the scan (path to the filename to scan */
	memset(scan, 0, sizeof(scan));
	memset(dirscan, 0, sizeof(dirscan));

	/* Clear the config_file */
	memset(config_file, 0, sizeof(config_file));

	/* Parse cmdline arguments */
	while ((c = getopt(argc, argv, "cC:Ddf:s:hv")) != EOF)
	{
		switch(c)
		{
			case 'c':
			sophie_config_reset();
			config_ret = sophie_load_config(config_file);
			if (config_ret != 1)
				exit(EXIT_FAILURE);
			sophie_init();
			sophie_show_settings();
			sophie_end();
			exit(0);
			break;
			
			case 'C':
			if (optarg != '\0')
				strncpy(config_file, optarg, sizeof(config_file)-1);
			break;

			case 'D':
			SOPHIE_DAEMON = 1;
			break;
			
			case 'd':
			SOPHIE_DEBUG = 1;
			break;

			case 'f':
			if (optarg != '\0')
				strncpy(scan, optarg, sizeof(scan)-1);
			break;
			
			case's':
			if (optarg != '\0')
				strncpy(dirscan, optarg, sizeof(dirscan)-1);
			break;

			case 'h':
			usage();
			exit(0);
			break;

			case 'v':
			SOPHIE_DAEMON = 0;
			SOPHIE_DEBUG = 0;
			sophie_config_reset();
			config_ret = sophie_load_config(config_file);
			if (config_ret != 1)
				exit(EXIT_FAILURE);
			sophie_init();
			sophie_version();
			sophie_end();
			exit(0);
			break;

			default:
			SOPHIE_DAEMON = 0;
			SOPHIE_DEBUG = 0;
			break;
		}
	}

	/* Reset config data */
	sophie_config_reset();
	
	/* Load Sophie (not SAVI) configuration file */
	config_ret = sophie_load_config(config_file);
	if (config_ret != 1)
		exit(EXIT_FAILURE);

	/* Make sure user really exists. Otherwise, Sophie will start, but childs will die if the user doesn't exist */
	if (!(checkUser = getpwnam(config.user)))
	{
		sophie_print(0, "%s config.user (%s) does not exist - exiting", ERRSTR, config.user);

		if (SOPHIE_DAEMON == 1)
			fprintf(stderr, "%s config.user (%s) does not exist - exiting\n", ERRSTR, config.user);

		exit(EXIT_FAILURE);
	}

	/* We will check if the config.group exists first */
	if (!getgrnam(config.group))
	{
		sophie_print(0, "%s config.group (%s) does not exist - exiting", ERRSTR, config.group);

		if (SOPHIE_DAEMON == 1)
			fprintf(stderr, "%s config.group (%s) does not exist - exiting\n", ERRSTR, config.group);

		exit(EXIT_FAILURE);
	}

	/* If SOPHIE_DEBUG/SOPHIE_DAEMON are not set (0 or 1) = set to 0 */
	if (SOPHIE_DEBUG != 0 && SOPHIE_DEBUG != 1)
		SOPHIE_DEBUG = 0;
	
	if (SOPHIE_DAEMON != 0 && SOPHIE_DAEMON != 1)
		SOPHIE_DAEMON = 0;	

	/* If file is specified on a command line, scan it and return result */
	/* When scan is finished, exit */
	if (scan[0] != '\0')
	{
        sophie_init();
        sophie_version();
		sophie_print(0, "%s Scanning file '%s' ", NOTESTR, scan);
		sophos_result = sophie_scanfile(scan);
		sophie_end();
		sophie_print(0, "%s SAVI cleaned up and terminated", NOTESTR);
		exit(EXIT_SUCCESS);
	}
	
	/* Yes, I could combine the above with this, but for the time being I want to have them separated */
	if (dirscan[0] != '\0')
	{
        sophie_init();
        sophie_version();
		sophie_print(0, "%s Scanning dir '%s'", NOTESTR, dirscan);
		sophos_result = sophie_scandir(dirscan);
		sophie_end();
		sophie_print(0, "%s SAVI cleaned up and terminated", NOTESTR);
		exit(EXIT_SUCCESS);
	}


	/* Check if the socket already exists */
	if (stat(config.socketfile, &fileInfo) == 0)
	{
		if (S_ISSOCK(fileInfo.st_mode))
		{
			sophie_print(0, "%s Socket '%s' already exists!", WARNSTR, config.socketfile);
			sophie_print(0, "%s Removing existing socket", NOTESTR);
			if (unlink(config.socketfile) == -1)
			{
				sophie_print(0, "%s Could not remove socket '%s' [%s]. Aborting!", ERRSTR, config.socketfile, strerror(errno));
				exit(EXIT_FAILURE);
			}
		}
	}

	/* Create a socket */
	if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) == -1)
	{
		sophie_print(0, "%s socket() failed in main() [%s]", ERRSTR, strerror(errno));
		exit(EXIT_FAILURE);
	}

	/* Set the umask */
	umask(config.umask);
	
	/* Set the socket properties (family, name) */
	server.sun_family = AF_UNIX;
	strncpy(server.sun_path, config.socketfile, sizeof(server.sun_path)-1);
/*	sophie_print(1, "%s config.socketfile is at '%s'", NOTESTR, config.socketfile); */

	/* Bind a socket */
	if (bind(sock, (struct sockaddr *) &server, sizeof(struct sockaddr_un)) == -1)
	{
		sophie_print(0, "%s bind() failed in main() [%s]", ERRSTR, strerror(errno));
		exit(EXIT_FAILURE);
	}

	/* Change the owner/group of the socket */
	grpInfo = getgrnam(config.group);
	if (chown(config.socketfile, checkUser->pw_uid, grpInfo->gr_gid) == -1)
		sophie_print(0, "%s failed to chown() the socket (user: %s, group %s) [%s]", WARNSTR, config.user, config.group, strerror(errno));

	/* Start listening */
	if (listen(sock, config.maxproc) == -1)
	{
		sophie_print(0, "%s listen() failed in main() [%s]", ERRSTR, strerror(errno));
		exit(EXIT_FAILURE);
	}

	/* If SOPHIE_DAEMON is set, show some message */
	/* Argh - at this point, everything already goes to syslog (if SOPHIE_DAEMON is set) - will change this later :) */
	if (SOPHIE_DAEMON == 1)
	{
		if ((pid = fork()) > 0)
		{
			sophie_print(0, "%s Placed in the background [PID: %d]", program_name, pid);
			exit(EXIT_SUCCESS);
		}
		else if (pid == -1)
		{
/*			perror("fork"); */
			sophie_print(0, "%s Could not fork() - aborting [%s]", ERRSTR, strerror(errno));
			exit(EXIT_FAILURE);
		}

		if (setsid() == -1)
		{
/*			perror("setsid");*/
			sophie_print(0, "%s Could not set process group() - aborting [%s]", ERRSTR, strerror(errno));
			exit(EXIT_FAILURE);
		}
		if ((devnull = open("/dev/null", O_RDWR, 0)) < 0)
		{
/*			perror("open");*/
			sophie_print(0, "%s Could not detach fds - aborting [%s]", ERRSTR, strerror(errno));
			exit(EXIT_FAILURE);
		}

		fprintf(stderr, "%s placed in the background\n", program_name);
		(void)dup2(devnull, STDIN_FILENO);
		(void)dup2(devnull, STDOUT_FILENO);
		(void)dup2(devnull, STDERR_FILENO);
		if (devnull > 2)
			(void)close (devnull);
	}	

#ifdef SOPHIE_NET
	if ((proto = getprotobyname("tcp")) == NULL)
	{
		sophie_print(0, "%s Could not get protocol number for TCP [%s]", ERRSTR, strerror(errno));
		exit(EXIT_FAILURE);
	}

	if ((tcp_sock = socket(AF_INET, SOCK_STREAM, proto->p_proto)) < 0)
	{
		sophie_print(0, "%s Could not obtain TCP socket [%s]", ERRSTR, strerror(errno));
		exit(EXIT_FAILURE);
	}

	bzero((char *) &tcp_sock_data, sizeof(tcp_sock_data));
	tcp_sock_data.sin_family = AF_INET;
	tcp_sock_data.sin_addr.s_addr = htonl(INADDR_ANY); /* FIXME: configurable? */
	tcp_sock_data.sin_port = htons(config.net_port);

	setsockopt(tcp_sock, SOL_SOCKET, SO_REUSEADDR, &reuse_addr, sizeof(reuse_addr));

	if (bind(tcp_sock, (struct sockaddr *) &tcp_sock_data, sizeof(tcp_sock_data)) < 0)
	{
		sophie_print(0, "%s Could not bind local socket [%s]", ERRSTR, strerror(errno));
/*		perror("Could not bind local socket\n");*/
		exit(EXIT_FAILURE);
	}

	listen(tcp_sock, config.maxproc);

	/* Make sockets non-blocking */
	fcntl(sock, F_SETFL, O_NONBLOCK);
	fcntl(tcp_sock, F_SETFL, O_NONBLOCK);
#endif

    /* Fork main thread that runs as non-root user.  We init SAVI as 
       this user because we must init SAVI as the same user as we scan
       with SAVI. */
    cpid = fork();
    if (cpid == -1) {
        sophie_print(0, "%s main fork() failed! [%s]", ERRSTR, strerror(errno));
        cleanup();    
        exit(EXIT_FAILURE);
    }
    else if (cpid > 0) {
        /* We are the startup process.  Now all we need to do let the main process
           perform the bulk of the work as the non-root user.  We just wait for
           it to finish, and cleanup our socket and pidfile when it does. */
        int mstatus;        
        
        /* Create PID file with pid of working process*/
        f = fopen(config.pidfile, "wb");
        if (f) {
            fprintf(f, "%u\n", (u_int) cpid);
            fclose(f);
        }
		wpid = cpid;

		signal(SIGALRM, SIG_IGN);
		signal(SIGUSR1, sig_main_general);
		signal(SIGHUP, sig_main_general);
		signal(SIGINT, sig_main_general);
		signal(SIGTERM, sig_main_general);
		signal(SIGQUIT, sig_main_general);
		signal(SIGSEGV, sig_main_general);

        waitpid(cpid, &mstatus, 0);        
        cleanup();
        exit(0);
    }        

    /* Only the main working process gets here.  
       Switch to non-root user and install signal handlers. */

    /* We do setuid() after PIDFILE was created (/var/run is owned by root) */
	userInfo = getpwnam(config.user);

	/* Check if the userInfo->pw_gid is valid (it should never be NULL, but...) */
	if (userInfo->pw_gid)
	{
		if ((setgid(userInfo->pw_gid)) == -1)
			sophie_print(0, "%s Failed to setgid() to '%d' [%s]", WARNSTR, userInfo->pw_gid, strerror(errno));
		else
			sophie_print(1, "%s Successful setgid() to '%d'", NOTESTR, userInfo->pw_gid);
	}
	else
	{
		sophie_print(1, "%s userInfo->pwgid is not valid", WARNSTR);
	}

	/* Init supplementary group access list*/
	/* Seems like only root can use this? */
	if (getuid() == 0)
	{
		if (initgroups(config.user, userInfo->pw_gid) == -1)
		{
			sophie_print(0, "%s initgroups() failed [%s]", ERRSTR, strerror(errno));
			exit(EXIT_FAILURE);
		}
	}

	if ((setuid(userInfo->pw_uid)) == -1)
		sophie_print(0, "%s Failed to setuid() to '%d' [%s]", WARNSTR, userInfo->pw_uid, strerror(errno));
	else
		sophie_print(1, "%s Successful setuid() to '%d'", NOTESTR, userInfo->pw_uid);
    


    sigemptyset(&set);
	sigprocmask(SIG_SETMASK, &set, NULL);

	signal(SIGALRM, SIG_IGN);
/*	signal(SIGCHLD, sig_chld); */
/*	signal(SIGHUP, sig_reload); */
	signal(SIGINT, sig_exit);
	signal(SIGTERM, sig_exit);
	signal(SIGQUIT, sig_exit);
	signal(SIGSEGV, sig_exit);
	sophie_print(1, "%s Signal handlers set", NOTESTR);

	/* SIGCHLD handler */
	sa_chld.sa_handler = sig_chld;
	sigfillset(&sa_chld.sa_mask);
	sa_chld.sa_flags = SA_RESTART;
	sig_ret = sigaction(SIGCHLD, &sa_chld, NULL);
	if (sig_ret < 0)
		sophie_print(0, "%s Could not install SIGCHLD signal handler (sig_ret = [%d])", ERRSTR, sig_ret);

	/* SIGHUP handler */
	sa_hup.sa_handler = sig_reload;
	sigfillset(&sa_hup.sa_mask);
	sa_hup.sa_flags = SA_RESTART;
	sig_ret = sigaction(SIGHUP, &sa_hup, NULL);
	if (sig_ret < 0)
		sophie_print(0, "%s Could not install SIGHUP signal handler (sig_ret = [%d])", ERRSTR, sig_ret);

	/* SIGUSR1 handler */
	sa_usr1.sa_handler = sig_usr1;
	sigfillset(&sa_usr1.sa_mask);
	sa_usr1.sa_flags = SA_RESTART;
	sig_ret = sigaction(SIGUSR1, &sa_usr1, NULL);
	if (sig_ret < 0)
		sophie_print(0, "%s Could not install SIGUSR1 signal handler (sig_ret = [%d])", ERRSTR, sig_ret);     


    /* init and config savi */
    sophie_init();
    sophie_version();        

	/* Fun begins... */
	for (;;)
	{
LOOP:
		/* Accept the connection(s) */
		nanosleep(&rqtp, NULL);
#ifdef SOPHIE_NET
		if ( ((msgsock = accept(sock, 0, 0)) > 0) || ((client_tcp_sock = accept(tcp_sock, &addr, &addrlen)) > 0) )
#else
		if ((msgsock = accept(sock, 0, 0)) > 0)
#endif
		{
			if (msgsock > 0)
				scan_type = SCAN_LOCAL;
#ifdef SOPHIE_NET
			if (client_tcp_sock > 0)
				scan_type = SCAN_NETWORK;
#endif

			sophie_print(1, "%s accept() set, scan type [%d]", NOTESTR, scan_type);

			/*
			   If we got EINTR, it means some signal is caught -
			   on HUP, we'll reload (and ignore this error), and on
			   other signal we'll die anyway (probably :)
			*/

			if ((scan_type == SCAN_LOCAL) && (msgsock == -1 && errno != EINTR))
			{
				sophie_print(0, "%s Error in local accept() [%s] - moving on", WARNSTR, strerror(errno));
			}
			else if (scan_type == SCAN_LOCAL && msgsock == -1) /* this should be EINTR now */
			{
				sophie_print(1, "%s EINTR set/caught in local accept() loop - some signal received", NOTESTR);
				continue;
			}
#ifdef SOPHIE_NET
			else if ((scan_type == SCAN_NETWORK) && (client_tcp_sock == -1 && errno != EINTR))
			{
				sophie_print(0, "%s Error in network accept() [%s] - moving on", WARNSTR, strerror(errno));
			}
			else if (scan_type == SCAN_NETWORK && client_tcp_sock == -1)
			{
				sophie_print(1, "%s EINTR set/caught in network accept() loop - some signal received", NOTESTR);
			}
#endif
			else
			{
				/* Suggested by philipp@corpex.de (Philipp Gaschtz) */
				while( (PROC_COUNT >= config.maxproc) && (config.maxproc > 0) )
					nanosleep(&rqtp, NULL);

/* ############################# */

			/* Fork */
			sigemptyset(&bset);
			sigaddset(&bset, SIGCHLD);
			sigprocmask(SIG_BLOCK, &bset, NULL);

			cpid = fork();

			if (cpid == -1)
			{
				sigprocmask(SIG_UNBLOCK, &bset, NULL);
				sophie_print(0, "%s fork() failed! [%s]", ERRSTR, strerror(errno));
				close(msgsock);
				goto LOOP;
			}
			else if (cpid > 0)
			{
				close(msgsock);
#ifdef SOPHIE_NET
				close(client_tcp_sock);
#endif
				PROC_COUNT++;
				sigprocmask(SIG_UNBLOCK, &bset, NULL);
				sophie_print(1, "%s fork()ed a child - everything seems ok", NOTESTR);
				continue;
			}
			else if (cpid == 0)
			{
				struct sophie_notification notification;
#ifdef USE_FGETS
				FILE *msgfp;
#else
				int ret;
#endif
				/* we are a chld */
				is_child = 1;

				/* child */
				sigprocmask(SIG_UNBLOCK, &bset, NULL);

				/* Print the PROC_COUNT */
				sophie_print(1, "%s Current PROC_COUNT is '%d'", NOTESTR, PROC_COUNT);

                /* We no longer need to switch to the non-root user after every fork now,
                   because we already are the non-root user.  The user switch code was cut 
                   from here and moved outside the main processing loop. */

				/*	bzero((char *)(&sa_exit), sizeof(sa_exit)); */
				sigemptyset(&sa_handle.sa_mask);
				sa_handle.sa_handler = sophie_timeout;

				if ((sig_ret = sigaction(SIGALRM, &sa_handle, NULL)) < 0)
					sophie_print(0, "%s Error setting handler for SIGALRM [%s]", WARNSTR, strerror(errno));

				/* Clean the buffer */				
				memset(buf, 0, sizeof(buf));
				sophie_print(1, "%s Cleared buf using memset()", NOTESTR);

#ifdef SOPHIE_NET
				if (scan_type == SCAN_NETWORK)
				{
					set_sockopts(client_tcp_sock);
					sophie_net_scan(client_tcp_sock);
/*					shutdown(client_tcp_sock, 2); */
					close(client_tcp_sock);
					_exit(EXIT_SUCCESS);
				}
				else
				{
#endif
					/*
					  We will set the alarm (timeout)
					  If Sophie doesn't receive anything within config.timeout seconds - quit
					*/
					alarm(config.timeout);

#ifdef USE_FGETS
					/* We will 'reopen' the socket, for fgets - I am lazy... :) */
					msgfp = fdopen(msgsock, "r");

					while (fgets(buf, sizeof(buf)-1, msgfp))
					{
						sophie_print(1, "%s read %d bytes from socket", NOTESTR, strlen(buf));
#else
					while ((ret = sophie_getline(buf, sizeof(buf), msgsock)))
					{
						sophie_print(1, "%s read %d bytes from socket", NOTESTR, ret);
#endif

						/* Clear the virus name */
						memset(VIR_NAME, 0, sizeof(VIR_NAME));

						/* Clear the ret_error_string */
						memset(ret_error_string, 0, sizeof(ret_error_string));

						/* Remove the newline (if it exists) */
						if (strchr(buf, '\n'))
							*strchr(buf, '\n') = '\0';
					
						/* Set the SIGPIPE handler */
						signal(SIGPIPE, sig_pipe);

						sophie_print(1, "%s Read: '%s'", NOTESTR, buf);
					
						/* Now, let's decide if we'll scan file or a dir... */
						if ((stat(buf, &reqstat)) == 0)
						{
							if (S_ISDIR(reqstat.st_mode))
							{
								sophos_result = sophie_scandir(buf);
							}
							else if (S_ISREG(reqstat.st_mode))
							{
								int not_ret;
								
								if (config.callbacks)
								{
									/* Register SAVI notification interface */
									notification.socket = msgsock;
									notification.interrupted = 0;
									memset(notification.message, 0, sizeof(notification.message));
								
									not_ret = SetNotification(pSAVI, (void *) &notification);
									if (not_ret != 1)
										sophie_print(1, "%s SetNotification() failed", WARNSTR);
									else
										sophie_print(1, "%s Callbacks set", NOTESTR);
								}

								sophos_result = sophie_scanfile(buf);

								/* This is needed because SweepFile() will
								   return SOPHOS_S_OK even if we've
								   interrupted it with SOPHOS_SAVI_CBCK_STOP
								   
								   We set interrupted to 1, message to actual
								   error message, and reconstruct the ret_error_string
								   
								   We also set sophos_result to -1 (so that scanned
								   file doesn't show up as 'okay'
								*/

								if (config.callbacks)
								{
									if (notification.interrupted == 1)
									{
										strncpy(ret_error_string, notification.message, sizeof(ret_error_string)-1);
										sophos_result = -1;
									}
								}

							}
							else
							{
								sophie_print(0, "%s What was requested is not file or directory", WARNSTR);
								snprintf(ret_error_string, sizeof(ret_error_string)-1, "'%s' not scanned (not a file/directory)", buf);
								sophos_result = -1;
							}
						}
						else
						{
							/* File/dir doesn't exist; permission denied? */
							sophie_print(0, "%s Error accessing '%s' (%s)", WARNSTR, buf, strerror(errno));
							snprintf(ret_error_string, sizeof(ret_error_string)-1, "%s (%s)", buf, strerror(errno));
							sophos_result = -1;
						}

						/* Clear the response buffer */
						memset(sock_response, 0, sizeof(sock_response));

						/* "Craft" the response (this is so lame) - FIXME */
						if ( (VIR_NAME[0] != '\0') && (config.show_virusname == 1) )
							snprintf(sock_response, sizeof(sock_response)-1, "%d:%s", sophos_result, VIR_NAME);
						else if ( (ret_error_string[0] != '\0') && (config.error_strings == 1) )
							snprintf(sock_response, sizeof(sock_response)-1, "%d:%s", sophos_result, ret_error_string);
						else
							snprintf(sock_response, sizeof(sock_response)-1, "%d", sophos_result);

						sophie_print(1, "%s Response is '%s'", NOTESTR, sock_response);
/*						sophie_print(1, "%s Sending the response '%s' to socket", NOTESTR, sock_response); */

						/* Complain on errors - do nothing if everything is okay */
						if (write(msgsock, sock_response, strlen(sock_response)) > 0)
						{
							sophie_print(1, "%s Response '%s' sent", NOTESTR, sock_response);
						}
						else
						{
							sophie_print(0, "%s write() to remote end of the socket failed! [%s]", WARNSTR, strerror(errno));
							if (notification.interrupted == 1)
								sophie_print(0, "%s write() failed because: %s", NOTESTR, notification.message);
						}

						/*
						   Before, we would setup alarm that affects the length of the whole 'communication' between
						   Sophie and the client. Now, we'll make alarm per file (input from the user), so that client
						   can keep (more or less) persistent connection.
						*/
						alarm(config.timeout);
						sophie_print(1, "%s Alarm (%d) set", NOTESTR, config.timeout);
					}

					sophie_print(1, "%s Child finished", NOTESTR);
				}

				alarm(0);

				/* End of the child process */
				_exit(EXIT_SUCCESS);
#ifdef SOPHIE_NET
			}
#endif
		}

		if (scan_type == SCAN_LOCAL)
			close(msgsock);

#ifdef SOPHIE_NET
		if (scan_type == SCAN_NETWORK)
			close(client_tcp_sock);
#endif
	}
	}
}
